<?php
session_start();
require_once '../config/db.php';
require_once '../helpers/helpers.php';
require_once '../includes/auth_check.php';
require_once '../includes/sms_functions.php';
require_once '../helpers/favicon_helper.php';
require_once '../includes/functions.php';  // Add this line to include functions.php
require_once '../helpers/permission_modal.php';

// Debug information
error_log("User ID: " . ($_SESSION['user_id'] ?? 'Not set'));
error_log("Username: " . ($_SESSION['username'] ?? 'Not set'));
error_log("Has view_settings permission: " . (hasPermission('view_settings') ? 'Yes' : 'No'));

// Check if user has any settings-related permissions
$settingsPermissions = [
    'view_settings', 'manage_store_settings', 'manage_email_settings', 'manage_mpesa_settings', 
    'manage_sms_settings', 'manage_whatsapp_settings', 'manage_users'
];

if (!hasAnyPermission($settingsPermissions)) {
    $_SESSION['error'] = "You don't have permission to access settings.";
    header('Location: dashboard.php');
    exit;
}

// Check if user has permission to manage specific settings
$canManageEmail = hasPermission('manage_email_settings');
$canManageMpesa = hasPermission('manage_mpesa_settings');
$canManageSms = hasPermission('manage_sms_settings');

// If user is Super Admin, grant all permissions
if (getUserRole() === 'Super Admin') {
    $canManageEmail = true;
    $canManageMpesa = true;
    $canManageSms = true;
}

$activePage = 'settings';

$success = '';
$error = '';

// Get all roles for the user creation form
$rolesForUser = $pdo->query("SELECT * FROM roles ORDER BY name")->fetchAll();

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Initialize credit settings with default values
$creditSettings = [
    'enable_credit_system' => '0',
    'default_credit_limit' => '1000',
    'max_credit_limit' => '10000',
    'credit_payment_terms' => '30',
    'late_payment_fee' => '5'
];

// Get credit settings from database and merge with defaults
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'credit_settings'");
$stmt->execute();
$dbSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
if (!empty($dbSettings)) {
    $creditSettings = array_merge($creditSettings, $dbSettings);
}

// Handle credit settings save
if (isset($_POST['save_credit_settings'])) {
    try {
        // Validate inputs
        $enableCreditSystem = isset($_POST['enable_credit_system']) ? '1' : '0';
        $defaultCreditLimit = floatval($_POST['default_credit_limit'] ?? $creditSettings['default_credit_limit']);
        $maxCreditLimit = floatval($_POST['max_credit_limit'] ?? $creditSettings['max_credit_limit']);
        $creditPaymentTerms = intval($_POST['credit_payment_terms'] ?? $creditSettings['credit_payment_terms']);
        $latePaymentFee = floatval($_POST['late_payment_fee'] ?? $creditSettings['late_payment_fee']);

        // Validate credit limits
        if ($defaultCreditLimit > $maxCreditLimit) {
            throw new Exception("Default credit limit cannot exceed maximum credit limit");
        }

        // Update settings
        $settings = [
            'enable_credit_system' => $enableCreditSystem,
            'default_credit_limit' => $defaultCreditLimit,
            'max_credit_limit' => $maxCreditLimit,
            'credit_payment_terms' => $creditPaymentTerms,
            'late_payment_fee' => $latePaymentFee
        ];

        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) 
                                 VALUES (?, ?, 'credit_settings') 
                                 ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
            $stmt->execute([$key, $value]);
        }

        $success = "Credit settings saved successfully!";
        
        // Refresh credit settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'credit_settings'");
        $stmt->execute();
        $dbSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        $creditSettings = array_merge($creditSettings, $dbSettings);
        
    } catch (Exception $e) {
        $error = "Error saving credit settings: " . $e->getMessage();
    }
}

// Handle add system user
if (
    isset($_POST['add_system_user']) &&
    hasPermission('manage_users') &&
    !empty($_POST['username']) &&
    !empty($_POST['password']) &&
    !empty($_POST['name']) &&
    !empty($_POST['email']) &&
    !empty($_POST['role_id'])
) {
    try {
        $stmt = $pdo->prepare("INSERT INTO users (username, password, name, email, role_id) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([
            $_POST['username'],
            password_hash($_POST['password'], PASSWORD_DEFAULT),
            $_POST['name'],
            $_POST['email'],
            $_POST['role_id']
        ]);
        $success = 'System user added successfully!';
    } catch (PDOException $e) {
        $error = 'Error adding system user: ' . $e->getMessage();
    }
}

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_settings'])) {
    try {
        // Get old settings for comparison
        $stmt = $pdo->query("SELECT * FROM settings");
        $old_settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        // Update settings
        $changes = [];
        foreach ($_POST['settings'] as $key => $value) {
            if (isset($old_settings[$key]) && $old_settings[$key] !== $value) {
                $stmt = $pdo->prepare("UPDATE settings SET value = ? WHERE setting_key = ?");
                if ($stmt->execute([$value, $key])) {
                    $changes[] = "$key: {$old_settings[$key]} → $value";
                }
            }
        }
        
        if (!empty($changes)) {
            $details = "Settings changed:\n" . implode("\n", $changes);
            logSystemAction('settings_updated', $details);
            $success = 'Settings updated successfully!';
        }
    } catch (PDOException $e) {
        $error = 'Error updating settings: ' . $e->getMessage();
    }
}

// Get all settings grouped by category
$stmt = $pdo->query("SELECT * FROM store_settings ORDER BY setting_group, setting_key");
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// M-Pesa Integration Settings
if (isset($_POST['save_mpesa_settings'])) {
    $enableMpesa = isset($_POST['enable_mpesa']) ? '1' : '0';
    $consumerKey = trim($_POST['mpesa_consumer_key'] ?? '');
    $consumerSecret = trim($_POST['mpesa_consumer_secret'] ?? '');
    $shortcode = trim($_POST['mpesa_shortcode'] ?? '');
    $passkey = trim($_POST['mpesa_passkey'] ?? '');
    
    // Validate required fields if M-Pesa is enabled
    if ($enableMpesa === '1') {
        if (empty($consumerKey) || empty($consumerSecret) || empty($shortcode) || empty($passkey)) {
            $error = 'All M-Pesa API credentials are required when M-Pesa is enabled.';
        } else {
            $settings = [
                'enable_mpesa' => $enableMpesa,
                'mpesa_consumer_key' => $consumerKey,
                'mpesa_consumer_secret' => $consumerSecret,
                'mpesa_shortcode' => $shortcode,
                'mpesa_passkey' => $passkey
            ];
            foreach ($settings as $key => $value) {
                $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) VALUES (?, ?, 'mpesa_settings') ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
                $stmt->execute([$key, $value]);
            }
            $success = 'M-Pesa settings updated successfully.';
        }
    } else {
        // If M-Pesa is disabled, just update the enable flag
        $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) VALUES ('enable_mpesa', '0', 'mpesa_settings') ON DUPLICATE KEY UPDATE setting_value = '0'");
        $stmt->execute();
        $success = 'M-Pesa payments have been disabled.';
    }
    
    // Refresh settings
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'mpesa_settings'");
    $stmt->execute();
    $storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
}

// Handle store information save
if (isset($_POST['save_store_settings'])) {
    try {
        // Get form data
        $storeName = trim($_POST['store_name'] ?? '');
        $storeEmail = trim($_POST['store_email'] ?? '');
        $storePhone = trim($_POST['store_phone'] ?? '');
        $storeAddress = trim($_POST['store_address'] ?? '');
        $website = trim($_POST['website'] ?? '');
        $currency = trim($_POST['currency'] ?? 'Ksh');

        // Add https:// to website if not present
        if (!empty($website)) {
            if (!str_starts_with($website, 'http://') && !str_starts_with($website, 'https://')) {
                $website = 'https://' . $website;
            }
        }

        // Handle logo upload
        if (isset($_FILES['store_logo']) && $_FILES['store_logo']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['store_logo'];
            $maxFileSize = 2 * 1024 * 1024; // 2MB
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
            
            // Validate file
            if ($file['size'] > $maxFileSize) {
                throw new Exception("Logo file size must be less than 2MB");
            }
            
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $file['tmp_name']);
            finfo_close($finfo);
            
            if (!in_array($mimeType, $allowedTypes)) {
                throw new Exception("Only JPG, PNG and GIF images are allowed");
            }
            
            // Read the file content
            $logoContent = file_get_contents($file['tmp_name']);
            if ($logoContent === false) {
                throw new Exception("Failed to read logo file");
            }

            // Save logo to database
            $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_binary, setting_group) 
                                 VALUES ('store_logo', ?, ?, 'store_info') 
                                 ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), setting_binary = VALUES(setting_binary)");
            $stmt->execute([$file['name'], $logoContent]);
        }

        // Handle favicon upload
        if (isset($_FILES['favicon']) && $_FILES['favicon']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['favicon'];
            $maxFileSize = 1 * 1024 * 1024; // 1MB
            $allowedTypes = ['image/x-icon', 'image/png'];
            
            // Validate file
            if ($file['size'] > $maxFileSize) {
                throw new Exception("Favicon file size must be less than 1MB");
            }
            
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $file['tmp_name']);
            finfo_close($finfo);
            
            if (!in_array($mimeType, $allowedTypes)) {
                throw new Exception("Only ICO and PNG files are allowed for favicon");
            }
            
            // Read the file content
            $faviconContent = file_get_contents($file['tmp_name']);
            if ($faviconContent === false) {
                throw new Exception("Failed to read favicon file");
            }

            // Save favicon to database
            $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_binary, setting_group) 
                                 VALUES ('favicon', ?, ?, 'store_info') 
                                 ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), setting_binary = VALUES(setting_binary)");
            $stmt->execute([$file['name'], $faviconContent]);
        }

        // Save other store settings
        $settings = [
            'store_name' => $storeName,
            'store_email' => $storeEmail,
            'store_phone' => $storePhone,
            'store_address' => $storeAddress,
            'website' => $website,
            'currency' => $currency
        ];

        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) 
                                 VALUES (?, ?, 'store_info') 
                                 ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
            $stmt->execute([$key, $value]);
        }

        $success = 'Store information saved successfully!';
        
        // Refresh settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'store_info'");
        $stmt->execute();
        $storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        $settings = array_merge($settings, $storeSettings);
        
    } catch (Exception $e) {
        $error = 'Error saving store information: ' . $e->getMessage();
        error_log("Error: " . $e->getMessage());
    }
}

// Handle payment settings save
if (isset($_POST['save_payment_settings'])) {
    try {
        $settings = [
            'default_cash_drawer' => $_POST['default_cash_drawer'] ?? 'main',
            'opening_balance' => $_POST['opening_balance'] ?? '0.00',
            'closing_balance' => $_POST['closing_balance'] ?? '0.00',
            'drawer_status' => isset($_POST['drawer_status']) ? 'open' : 'closed'
        ];

        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) VALUES (?, ?, 'payment_settings') ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), setting_group = 'payment_settings'");
            $stmt->execute([$key, $value]);
        }

        $success = 'Payment settings saved successfully!';
    } catch (Exception $e) {
        $error = 'Error saving payment settings: ' . $e->getMessage();
        error_log("Error: " . $e->getMessage());
    }
}

// Fetch store settings
$storeSettings = [];
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'payment_settings'");
    $stmt->execute();
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $storeSettings[$row['setting_key']] = $row['setting_value'];
    }
} catch (Exception $e) {
    error_log("Error fetching store settings: " . $e->getMessage());
}

// Debug output
error_log("Store Settings: " . print_r($storeSettings, true));

// Handle settings save
if (isset($_POST['save_settings'])) {
    try {
        // Get today's cash sales
        $today = date('Y-m-d');
        $stmt = $pdo->prepare("SELECT COALESCE(SUM(total_amount),0) as total_sales_amount FROM sales WHERE DATE(created_at) = ? AND payment_method = 'cash' AND status = 'completed'");
        $stmt->execute([$today]);
        $today_cash_sales = $stmt->fetchColumn();

        // Save basic settings
        $settings = [
            'default_cash_drawer' => $_POST['default_cash_drawer'] ?? 'main',
            'opening_balance' => number_format((float)($_POST['opening_balance'] ?? 0), 2, '.', ''),
            'drawer_status' => isset($_POST['drawer_status']) ? 'open' : 'closed'
        ];

        // Debug output
        error_log("Saving settings: " . print_r($settings, true));

        // Save new values
        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) VALUES (?, ?, 'payment_settings') ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), setting_group = 'payment_settings'");
            $stmt->execute([$key, $value]);
            
            // Debug output
            error_log("Saved setting: $key = $value");
        }

        // Log the changes
        logSystemAction('settings_updated', 'Updated settings: ' . json_encode($settings), 'settings');

        // If closing balance is set, record the end of day
        if (!empty($_POST['closing_balance'])) {
            // Save closing balance to settings
            $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) VALUES ('closing_balance', ?, 'payment_settings') ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), setting_group = 'payment_settings'");
            $stmt->execute([$_POST['closing_balance']]);

            // Record in cash drawer logs
            $stmt = $pdo->prepare("INSERT INTO cash_drawer_logs (opening_balance, closing_balance, total_sales, date, created_by) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([
                $_POST['opening_balance'],
                $_POST['closing_balance'],
                $today_cash_sales,
                $today,
                $_SESSION['user_id']
            ]);

            // Log the action
            logSystemAction('end_of_day', "End of day recorded. Opening: {$_POST['opening_balance']}, Closing: {$_POST['closing_balance']}, Sales: {$today_cash_sales}", 'cash_drawer');
        }

        // Refresh the page to show updated values
        header('Location: ' . $_SERVER['PHP_SELF'] . '?tab=payment&success=1');
        exit;
    } catch (Exception $e) {
        $error = 'Error saving settings: ' . $e->getMessage();
        error_log("Error saving settings: " . $e->getMessage());
    }
}

// Display success message if redirected after save
if (isset($_GET['success']) && $_GET['success'] == 1) {
    $success = 'Settings saved successfully!';
}

// Handle drawer reconciliation
if (isset($_POST['reconcile_drawer'])) {
    try {
        $actual_amount = $_POST['actual_amount'] ?? 0;
        $expected_amount = $_POST['expected_amount'] ?? 0;
        $difference = $actual_amount - $expected_amount;

        // Log the reconciliation
        logSystemAction('drawer_reconciliation', "Drawer reconciled. Expected: {$expected_amount}, Actual: {$actual_amount}, Difference: {$difference}");

        $success = 'Drawer reconciled successfully!';
    } catch (Exception $e) {
        $error = 'Error reconciling drawer: ' . $e->getMessage();
        error_log("Error: " . $e->getMessage());
    }
}

// Handle drawer history view
if (isset($_POST['view_history'])) {
    try {
        $start_date = $_POST['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
        $end_date = $_POST['end_date'] ?? date('Y-m-d');

        $stmt = $pdo->prepare("
            SELECT * FROM cash_drawer_logs 
            WHERE date BETWEEN ? AND ?
            ORDER BY date DESC
        ");
        $stmt->execute([$start_date, $end_date]);
        $drawer_history = $stmt->fetchAll();
    } catch (Exception $e) {
        $error = 'Error fetching drawer history: ' . $e->getMessage();
        error_log("Error: " . $e->getMessage());
    }
}

// Handle SMS test
if (isset($_POST['test_sms_settings'])) {
    $testResult = testSMSSettings();
    if ($testResult['success']) {
        $success = $testResult['message'];
    } else {
        $error = $testResult['message'];
    }
}

// Handle SMS settings save
if (isset($_POST['save_sms_settings'])) {
    try {
        $enableSMS = isset($_POST['enable_sms']) ? '1' : '0';
        $smsProvider = trim($_POST['sms_provider'] ?? '');
        
        // First, update the enable_sms setting
        $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) 
                             VALUES ('enable_sms', ?, 'sms_settings')
                             ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute([$enableSMS, $enableSMS]);

        if ($enableSMS === '1') {
            // Only validate and save SMS settings if SMS is enabled
        $settings = [
            'sms_provider' => $smsProvider,
            'sms_api_key' => trim($_POST['settings']['sms_api_key'] ?? ''),
            'sms_api_secret' => trim($_POST['settings']['sms_api_secret'] ?? ''),
            'sms_sender_id' => trim($_POST['settings']['sms_sender_id'] ?? ''),
            'sms_test_number' => trim($_POST['settings']['sms_test_number'] ?? '')
        ];
        
            // Validate required fields
            if (empty($smsProvider)) {
                throw new Exception('Please select an SMS provider');
            }
            if (empty($settings['sms_api_key']) || empty($settings['sms_api_secret']) || empty($settings['sms_sender_id'])) {
                throw new Exception('All SMS API credentials are required when SMS is enabled');
        }
        
        foreach ($settings as $key => $value) {
                $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) 
                                     VALUES (?, ?, 'sms_settings')
                                     ON DUPLICATE KEY UPDATE setting_value = ?");
                $stmt->execute([$key, $value, $value]);
            }
        }

        $success = 'SMS settings updated successfully!';

        // Refresh settings after save
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
        $stmt->execute();
        $smsSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        $settings = array_merge($settings, $smsSettings);
    } catch (Exception $e) {
        $error = 'Error updating SMS settings: ' . $e->getMessage();
    }
}

// Email Settings
if (isset($_POST['save_email_settings'])) {
    try {
    $enableEmail = isset($_POST['enable_email']) ? '1' : '0';
        
        // First, update the enable_email setting
        $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) 
                             VALUES ('enable_email', ?, 'email_settings')
                             ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute([$enableEmail, $enableEmail]);

    if ($enableEmail === '1') {
            // Only validate and save SMTP settings if email is enabled
            $smtpHost = $_POST['smtp_host'] ?? '';
            $smtpPort = $_POST['smtp_port'] ?? '';
            $smtpUsername = $_POST['smtp_username'] ?? '';
            $smtpPassword = $_POST['smtp_password'] ?? '';
            $smtpEncryption = $_POST['smtp_encryption'] ?? 'tls';
            $smtpSenderName = $_POST['smtp_sender_name'] ?? 'POS System';
            $emailTestAddress = $_POST['email_test_address'] ?? '';

            // Validate required fields
        if (empty($smtpHost) || empty($smtpPort) || empty($smtpUsername) || empty($smtpPassword)) {
                throw new Exception("All SMTP settings are required when email is enabled");
    }

        $settings = [
            'smtp_host' => $smtpHost,
            'smtp_port' => $smtpPort,
            'smtp_username' => $smtpUsername,
            'smtp_password' => $smtpPassword,
            'smtp_encryption' => $smtpEncryption,
            'smtp_sender_name' => $smtpSenderName,
            'email_test_address' => $emailTestAddress
        ];

        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) 
                                 VALUES (?, ?, 'email_settings')
                                 ON DUPLICATE KEY UPDATE setting_value = ?");
            $stmt->execute([$key, $value, $value]);
            }
        }

        $success = "Email settings updated successfully";
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Handle email test
if (isset($_POST['test_email'])) {
    require_once '../includes/email_functions.php';
    $result = testEmailSettings();
    if ($result['success']) {
        $success = $result['message'];
    } else {
        $error = $result['message'];
    }
}

// Get email settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'email_settings'");
$stmt->execute();
$emailSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Get SMS settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
$stmt->execute();
$smsSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Get M-Pesa settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'mpesa_settings'");
$stmt->execute();
$mpesaSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Get WhatsApp settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
$stmt->execute();
$whatsappSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Get Receipt settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'receipt_settings'");
$stmt->execute();
$receiptSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Merge all settings
$settings = array_merge($settings, $emailSettings, $smsSettings, $mpesaSettings, $whatsappSettings, $receiptSettings);

// Get favicon URL
$faviconUrl = getFaviconUrl($pdo);

// Handle WhatsApp settings save
if (isset($_POST['save_whatsapp_settings'])) {
    try {
    $enableWhatsapp = isset($_POST['enable_whatsapp']) ? '1' : '0';
    $whatsappMethod = trim($_POST['whatsapp_method'] ?? 'web_link');
    $whatsappSenderNumber = trim($_POST['whatsapp_sender_number'] ?? '');
    $whatsappApiKey = trim($_POST['whatsapp_api_key'] ?? '');
    $whatsappApiUrl = trim($_POST['whatsapp_api_url'] ?? '');
    $whatsappPhoneNumberId = trim($_POST['whatsapp_phone_number_id'] ?? '');
    $whatsappBusinessPhone = trim($_POST['whatsapp_business_phone'] ?? '');
    $whatsappTestNumber = trim($_POST['whatsapp_test_number'] ?? '');
    
    // Validate required fields if WhatsApp is enabled
    if ($enableWhatsapp === '1') {
        $requiredFields = [];
        
        // Check method-specific requirements
        if ($whatsappMethod === 'api') {
            if (empty($whatsappApiKey) || empty($whatsappApiUrl) || empty($whatsappPhoneNumberId)) {
                $requiredFields[] = 'API Key, API URL, and Phone Number ID';
            }
        } elseif ($whatsappMethod === 'business_app') {
            if (empty($whatsappBusinessPhone)) {
                $requiredFields[] = 'Business Phone Number';
            }
        }
        
        // Test number is always required
        if (empty($whatsappTestNumber)) {
            $requiredFields[] = 'Test Phone Number';
        }
        
        if (!empty($requiredFields)) {
                throw new Exception('Required fields for ' . ucfirst($whatsappMethod) . ' method: ' . implode(', ', $requiredFields));
            }
            
            // Save all settings if validation passes
            $settingsToSave = [
                'enable_whatsapp' => $enableWhatsapp,
                'whatsapp_method' => $whatsappMethod,
                'whatsapp_sender_number' => $whatsappSenderNumber,
                'whatsapp_api_key' => $whatsappApiKey,
                'whatsapp_api_url' => $whatsappApiUrl,
                'whatsapp_phone_number_id' => $whatsappPhoneNumberId,
                'whatsapp_business_phone' => $whatsappBusinessPhone,
                'whatsapp_test_number' => $whatsappTestNumber
            ];
            
            foreach ($settingsToSave as $key => $value) {
                $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) VALUES (?, ?, 'whatsapp_settings') ON DUPLICATE KEY UPDATE setting_value = ?");
                $stmt->execute([$key, $value, $value]);
            }
            $success = 'WhatsApp settings saved successfully!';
            
    } else {
        // If WhatsApp is disabled, just update the enable flag
        $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) VALUES ('enable_whatsapp', '0', 'whatsapp_settings') ON DUPLICATE KEY UPDATE setting_value = '0'");
        $stmt->execute();
        $success = 'WhatsApp notifications have been disabled.';
    }
    
        // Refresh settings after save
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
        $stmt->execute();
        $whatsappSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        // Merge with existing settings
        $settings = array_merge($settings, $whatsappSettings);
        
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Handle Receipt settings save
if (isset($_POST['save_receipt_settings'])) {
    $enableUnpaidCreditSection = isset($_POST['enable_unpaid_credit_section']) ? '1' : '0';
    $stmt = $pdo->prepare("INSERT INTO store_settings (setting_key, setting_value, setting_group) VALUES ('enable_unpaid_credit_section', ?, 'receipt_settings') ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
    $stmt->execute([$enableUnpaidCreditSection]);
    $success = 'Receipt settings saved successfully!';
    // Refresh settings
    $stmt = $pdo->query("SELECT * FROM store_settings ORDER BY setting_group, setting_key");
    $settings = [];
    while ($row = $stmt->fetch()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - POS System</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        :root {
            --primary-gradient: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            --success-gradient: linear-gradient(135deg, #10b981 0%, #059669 100%);
            --warning-gradient: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
            --danger-gradient: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            --card-shadow: 0 8px 30px rgba(37,99,235,0.12);
            --hover-shadow: 0 12px 40px rgba(37,99,235,0.15);
        }

        body { 
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); 
            font-family: 'Inter', sans-serif; 
            min-height: 100vh;
        }

        .card { 
            border-radius: 24px; 
            box-shadow: var(--card-shadow);
            border: none;
            background: rgba(255, 255, 255, 0.98);
            backdrop-filter: blur(10px);
            margin-bottom: 2rem;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            overflow: hidden;
        }

        .card:hover {
            transform: translateY(-2px);
            box-shadow: var(--hover-shadow);
        }

        .card-header { 
            background: none; 
            border-bottom: 1px solid rgba(37,99,235,0.1);
            padding: 2rem;
        }

        .card-title {
            font-size: 1.75rem;
            font-weight: 800;
            color: #1e293b;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 1rem;
            letter-spacing: -0.5px;
        }

        .card-title i {
            color: #2563eb;
            font-size: 2rem;
            background: var(--primary-gradient);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .btn {
            border-radius: 16px;
            font-weight: 700;
            padding: 0.875rem 1.75rem;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            font-size: 0.875rem;
            position: relative;
            overflow: hidden;
        }

        .btn::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(rgba(255,255,255,0.1), rgba(255,255,255,0));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .btn:hover::after {
            opacity: 1;
        }

        .btn-primary {
            background: var(--primary-gradient);
            border: none;
            box-shadow: 0 4px 15px rgba(37,99,235,0.2);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37,99,235,0.3);
        }

        .form-label { 
            font-weight: 700; 
            color: #1e293b;
            margin-bottom: 0.75rem;
            font-size: 0.95rem;
            letter-spacing: -0.25px;
        }

        .form-control {
            border-radius: 16px;
            border: 2px solid #e2e8f0;
            padding: 1rem 1.5rem;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            font-size: 0.95rem;
            background: rgba(255, 255, 255, 0.9);
        }

        .form-control:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 4px rgba(37,99,235,0.1);
            background: #fff;
        }

        .alert {
            border-radius: 20px;
            border: none;
            padding: 1.5rem 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
            backdrop-filter: blur(10px);
        }

        .alert-success {
            background: var(--success-gradient);
            color: white;
        }

        .alert-danger {
            background: var(--danger-gradient);
            color: white;
        }

        .settings-group {
            background: rgba(248, 250, 252, 0.8);
            border-radius: 20px;
            padding: 2.5rem;
            margin-bottom: 2rem;
            border: 1px solid rgba(37,99,235,0.1);
            backdrop-filter: blur(10px);
            transition: all 0.3s ease;
        }

        .settings-group:hover {
            background: rgba(248, 250, 252, 0.95);
            transform: translateY(-2px);
            box-shadow: 0 8px 30px rgba(37,99,235,0.08);
        }

        .settings-group-title {
            font-weight: 800;
            color: #1e293b;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 1rem;
            font-size: 1.25rem;
            letter-spacing: -0.5px;
        }

        .settings-group-title i {
            color: #2563eb;
            font-size: 1.5rem;
            background: var(--primary-gradient);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .form-check {
            padding: 1.25rem 1.5rem;
            border-radius: 16px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            margin-bottom: 1rem;
            background: white;
            border: 2px solid #e2e8f0;
            cursor: pointer;
        }

        .form-check:hover {
            background-color: #f8fafc;
            border-color: #2563eb;
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(37,99,235,0.08);
        }

        .form-check-input:checked {
            background-color: #2563eb;
            border-color: #2563eb;
        }

        .form-check-label {
            font-weight: 700;
            color: #1e293b;
            font-size: 1rem;
        }

        .nav-tabs {
            border: none;
            margin-bottom: 2.5rem;
            gap: 1rem;
            padding: 0.75rem;
            background: rgba(255, 255, 255, 0.5);
            border-radius: 20px;
            backdrop-filter: blur(10px);
            box-shadow: 0 4px 15px rgba(37,99,235,0.05);
        }

        .nav-tabs .nav-link {
            border: none;
            border-radius: 16px;
            padding: 1.25rem 1.75rem;
            font-weight: 700;
            color: #64748b;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            font-size: 1rem;
            letter-spacing: -0.25px;
        }

        .nav-tabs .nav-link:hover {
            background: rgba(37,99,235,0.08);
            color: #2563eb;
            transform: translateY(-2px);
        }

        .nav-tabs .nav-link.active {
            background: var(--primary-gradient);
            color: white;
            box-shadow: 0 4px 15px rgba(37,99,235,0.2);
        }

        .nav-tabs .nav-link i {
            margin-right: 0.75rem;
            font-size: 1.25rem;
        }

        .settings-description {
            font-size: 0.875rem;
            color: #64748b;
            margin-top: 0.75rem;
            font-weight: 500;
            line-height: 1.5;
        }

        .settings-icon {
            width: 56px;
            height: 56px;
            border-radius: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.75rem;
            color: white;
            margin-right: 1.5rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }

        .settings-icon.store {
            background: var(--primary-gradient);
        }

        .settings-icon.payment {
            background: var(--success-gradient);
        }

        .settings-icon.credit {
            background: var(--warning-gradient);
        }

        .settings-icon.system {
            background: var(--danger-gradient);
        }

        .table {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 4px 20px rgba(0,0,0,0.05);
        }

        .table th {
            background: #f8fafc;
            font-weight: 700;
            color: #1e293b;
            border-bottom: 2px solid #e2e8f0;
            padding: 1.5rem;
            font-size: 0.95rem;
            letter-spacing: -0.25px;
        }

        .table td {
            padding: 1.5rem;
            vertical-align: middle;
            color: #475569;
            font-size: 0.95rem;
        }

        .table-striped > tbody > tr:nth-of-type(odd) {
            background-color: #f8fafc;
        }

        .table-striped > tbody > tr:hover {
            background-color: #f1f5f9;
            transform: scale(1.01);
            transition: all 0.3s ease;
        }

        .badge {
            padding: 0.5rem 1rem;
            border-radius: 12px;
            font-weight: 700;
            font-size: 0.875rem;
            letter-spacing: -0.25px;
        }

        .badge.bg-success {
            background: var(--success-gradient) !important;
        }

        .badge.bg-danger {
            background: var(--danger-gradient) !important;
        }

        .btn-group .btn {
            padding: 0.75rem 1.25rem;
            border-radius: 12px;
        }

        .btn-group .btn i {
            font-size: 1rem;
        }

        .modal-content {
            border-radius: 24px;
            border: none;
            box-shadow: 0 8px 30px rgba(0,0,0,0.1);
            background: rgba(255, 255, 255, 0.98);
            backdrop-filter: blur(10px);
        }

        .modal-header {
            border-bottom: 1px solid rgba(37,99,235,0.1);
            padding: 2rem;
        }

        .modal-body {
            padding: 2rem;
        }

        .form-select {
            border-radius: 16px;
            border: 2px solid #e2e8f0;
            padding: 1rem 1.5rem;
            font-size: 0.95rem;
            background: rgba(255, 255, 255, 0.9);
        }

        .form-select:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 4px rgba(37,99,235,0.1);
            background: #fff;
        }

        .input-group-text {
            background: #f8fafc;
            border: 2px solid #e2e8f0;
            color: #64748b;
            font-weight: 700;
            padding: 1rem 1.5rem;
            border-radius: 16px;
            font-size: 0.95rem;
        }

        .drawer-status {
            display: inline-flex;
            align-items: center;
            padding: 0.5rem 1rem;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.875rem;
            margin-left: 1rem;
        }

        .drawer-status.open {
            background: var(--success-gradient);
            color: white;
        }

        .drawer-status.closed {
            background: var(--danger-gradient);
            color: white;
        }

        .drawer-actions {
            display: flex;
            gap: 1rem;
            margin-top: 1rem;
        }

        .drawer-amount {
            font-size: 1.5rem;
            font-weight: 800;
            color: #1e293b;
            margin: 1rem 0;
        }

        .drawer-history {
            margin-top: 2rem;
            padding-top: 2rem;
            border-top: 1px solid rgba(37,99,235,0.1);
        }

        .drawer-history-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1rem;
            border-radius: 12px;
            background: white;
            margin-bottom: 0.5rem;
            transition: all 0.3s ease;
        }

        .drawer-history-item:hover {
            transform: translateX(5px);
            box-shadow: 0 4px 15px rgba(37,99,235,0.08);
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                <div class="card mb-4">
         
                <div class="card">
                    <div class="card-header">
                        <h4 class="mb-0">System Settings</h4>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success"><?php echo $success; ?></div>
                        <?php endif; ?>
                        <?php if ($error): ?>
                            <div class="alert alert-danger"><?php echo $error; ?></div>
                        <?php endif; ?>

                        <ul class="nav nav-tabs mb-4" id="settingsTabs" role="tablist">
                            <li class="nav-item">
                                <a class="nav-link active" id="store-tab" data-bs-toggle="tab" href="#store" role="tab">
                                    <i class="fas fa-store me-2"></i>Store Information
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="credit-settings-tab" data-bs-toggle="tab" href="#credit-settings" role="tab">
                                    <i class="fas fa-credit-card me-2"></i>Credit Settings
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="email-tab" data-bs-toggle="tab" href="#email" role="tab">
                                    <i class="fas fa-envelope me-2"></i>Email Settings
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="mpesa-tab" data-bs-toggle="tab" href="#mpesa" role="tab">
                                    <i class="fas fa-mobile-alt me-2"></i>M-Pesa Settings
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="sms-tab" data-bs-toggle="tab" href="#sms" role="tab">
                                    <i class="fas fa-sms me-2"></i>SMS Settings
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="password-tab" data-bs-toggle="tab" href="#password" role="tab">
                                    <i class="fas fa-key me-2"></i>Password Management
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="payment-tab" data-bs-toggle="tab" href="#payment" role="tab">
                                    <i class="fas fa-credit-card me-2"></i>Payment Settings
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="whatsapp-tab" data-bs-toggle="tab" href="#whatsapp" role="tab">
                                    <i class="fab fa-whatsapp me-2"></i>WhatsApp Settings
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="receipt-tab" data-bs-toggle="tab" href="#receipt" role="tab">
                                    <i class="fas fa-receipt me-2"></i>Toggle for Receipts
                                </a>
                            </li>
                        </ul>

                        <div class="tab-content" id="settingsTabContent">
                            <!-- Store Information -->
                            <div class="tab-pane fade show active" id="store" role="tabpanel">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0">Store Information</h5>
                                    </div>
                                    <div class="card-body">
                                        <form method="post" enctype="multipart/form-data">
                                        <div class="row">
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Store Name</label>
                                                        <input type="text" class="form-control" name="store_name" 
                                                               value="<?php echo htmlspecialchars($settings['store_name'] ?? ''); ?>" required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Store Email</label>
                                                        <input type="email" class="form-control" name="store_email" 
                                                               value="<?php echo htmlspecialchars($settings['store_email'] ?? ''); ?>" required>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Store Phone</label>
                                                        <input type="text" class="form-control" name="store_phone" 
                                                               value="<?php echo htmlspecialchars($settings['store_phone'] ?? ''); ?>" required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Store Address</label>
                                                        <input type="text" class="form-control" name="store_address" 
                                                               value="<?php echo htmlspecialchars($settings['store_address'] ?? ''); ?>" required>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Website</label>
                                                        <div class="input-group">
                                                            <span class="input-group-text">https://</span>
                                                            <input type="text" class="form-control" name="website" 
                                                                   value="<?php echo htmlspecialchars(str_replace(['http://', 'https://'], '', $settings['website'] ?? '')); ?>"
                                                                   placeholder="www.example.com">
                                                        </div>
                                                        <small class="form-text text-muted">Enter your website without http:// or https://</small>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Currency</label>
                                                        <select class="form-select" name="currency" required>
                                                            <option value="Ksh" <?php echo ($settings['currency'] ?? '') === 'Ksh' ? 'selected' : ''; ?>>Kenyan Shilling (Ksh)</option>
                                                            <option value="USD" <?php echo ($settings['currency'] ?? '') === 'USD' ? 'selected' : ''; ?>>US Dollar ($)</option>
                                                            <option value="EUR" <?php echo ($settings['currency'] ?? '') === 'EUR' ? 'selected' : ''; ?>>Euro (€)</option>
                                                            <option value="GBP" <?php echo ($settings['currency'] ?? '') === 'GBP' ? 'selected' : ''; ?>>British Pound (£)</option>
                                                        </select>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Store Logo</label>
                                                        <input type="file" class="form-control" name="store_logo" accept="image/*">
                                                <?php
                                                        // Get logo from database
                                                        $stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info'");
                                                        $stmt->execute();
                                                        $logo = $stmt->fetch();
                                                        if ($logo && $logo['setting_binary']) {
                                                    $mimeType = 'image/jpeg'; // Default to JPEG
                                                            if (strpos($logo['setting_value'], '.png') !== false) {
                                                        $mimeType = 'image/png';
                                                            } elseif (strpos($logo['setting_value'], '.gif') !== false) {
                                                        $mimeType = 'image/gif';
                                                    }
                                                            echo '<div class="mt-2">';
                                                            echo '<img src="data:' . $mimeType . ';base64,' . base64_encode($logo['setting_binary']) . '" alt="Current Logo" class="img-thumbnail" style="max-height: 100px;">';
                                                            echo '</div>';
                                                }
                                                ?>
                                            </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Favicon</label>
                                                        <input type="file" class="form-control" name="favicon" accept="image/x-icon,image/png">
                                                <?php
                                                        // Get favicon from database
                                                        $stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'favicon' AND setting_group = 'store_info'");
                                                        $stmt->execute();
                                                        $favicon = $stmt->fetch();
                                                        if ($favicon && $favicon['setting_binary']) {
                                                    $mimeType = 'image/x-icon'; // Default to ICO
                                                            if (strpos($favicon['setting_value'], '.png') !== false) {
                                                        $mimeType = 'image/png';
                                                    }
                                                            echo '<div class="mt-2">';
                                                            echo '<img src="data:' . $mimeType . ';base64,' . base64_encode($favicon['setting_binary']) . '" alt="Current Favicon" class="img-thumbnail" style="max-height: 32px;">';
                                                            echo '</div>';
                                                }
                                                ?>
                                            </div>
        </div>
    </div>
                                            <button type="submit" name="save_store_settings" class="btn btn-primary" data-permission="manage_settings" data-permission-label="Manage Settings">
                                                <i class="fas fa-save"></i> Save Store Information
                                            </button>
                                    </form>
                                    </div>
                                </div>
                            </div>

                            <!-- Credit System Settings -->
                            <div class="tab-pane fade" id="credit-settings" role="tabpanel">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0">Credit System Settings</h5>
                                    </div>
                                    <div class="card-body">
                                        <form method="post" id="creditSettingsForm">
                                            <div class="form-check form-switch mb-3">
                                                <input class="form-check-input" type="checkbox" id="enableCreditSystem" name="enable_credit_system" value="1" 
                                                       <?php if (($settings['enable_credit_system'] ?? '0') == '1') echo 'checked'; ?>>
                                                <label class="form-check-label" for="enableCreditSystem">Enable Credit System</label>
                                                </div>
                                            <div id="creditSettings" class="<?php echo ($settings['enable_credit_system'] ?? '0') == '1' ? '' : 'd-none'; ?>">
                                                <div class="row">
                                                    <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Default Credit Limit</label>
                                                <div class="input-group">
                                                    <span class="input-group-text"><?php echo $currency; ?></span>
                                                    <input type="number" class="form-control" name="default_credit_limit" 
                                                                       value="<?php echo htmlspecialchars($settings['default_credit_limit'] ?? '1000'); ?>"
                                                           min="0" step="0.01" required>
                                                </div>
                                                            <small class="form-text text-muted">Default credit limit for new customers</small>
                                            </div>
                                                    </div>
                                                    <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Maximum Credit Limit</label>
                                                <div class="input-group">
                                                    <span class="input-group-text"><?php echo $currency; ?></span>
                                                    <input type="number" class="form-control" name="max_credit_limit" 
                                                                       value="<?php echo htmlspecialchars($settings['max_credit_limit'] ?? '10000'); ?>"
                                                           min="0" step="0.01" required>
                                                </div>
                                                            <small class="form-text text-muted">Maximum allowed credit limit</small>
                                            </div>
                                                    </div>
                                                </div>
                                                <div class="row">
                                                    <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Credit Payment Terms (Days)</label>
                                                <input type="number" class="form-control" name="credit_payment_terms" 
                                                                   value="<?php echo htmlspecialchars($settings['credit_payment_terms'] ?? '30'); ?>"
                                                                   min="1" required>
                                                            <small class="form-text text-muted">Number of days allowed for credit payment</small>
                                            </div>
                                                    </div>
                                                    <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Late Payment Fee (%)</label>
                                                <div class="input-group">
                                                    <input type="number" class="form-control" name="late_payment_fee" 
                                                                       value="<?php echo htmlspecialchars($settings['late_payment_fee'] ?? '5'); ?>"
                                                                       min="0" step="0.01" required>
                                                    <span class="input-group-text">%</span>
                                                </div>
                                                            <small class="form-text text-muted">Percentage charged for late payments</small>
                                            </div>
                                            </div>
                                                </div>
                                            </div>
                                            <button type="submit" name="save_credit_settings" class="btn btn-success" data-permission="manage_settings" data-permission-label="Manage Settings">Save Credit Settings</button>
                                        </form>
                                    </div>
                                </div>
                            </div>

                            <!-- Email Settings -->
                            <div class="tab-pane fade" id="email" role="tabpanel">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0">Email Settings</h5>
                                    </div>
                                    <div class="card-body">
                                        <form method="post" action="">
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" id="enable_email" name="enable_email" 
                                                           <?php echo ($settings['enable_email'] ?? '0') === '1' ? 'checked' : ''; ?>>
                                                    <label class="form-check-label" for="enable_email">Enable Email Notifications</label>
                                                </div>
                                            </div>

                                            <div id="emailSettings" class="<?php echo ($settings['enable_email'] ?? '0') === '1' ? '' : 'd-none'; ?>">
                                                <!-- Email Configuration Guide -->
                                                <div class="alert alert-info mb-4">
                                                    <h6 class="alert-heading"><i class="fas fa-info-circle"></i> Email Configuration Guide</h6>
                                                    <hr>
                                                    <div class="row">
                                                        <!-- Left Column -->
                                                        <div class="col-md-6">
                                                            <div class="mb-3">
                                                                <strong>Common SMTP Settings:</strong>
                                                                <ul class="mb-0">
                                                                    <li><strong>Gmail:</strong>
                                                                        <ul>
                                                                            <li>SMTP Host: smtp.gmail.com</li>
                                                                            <li>SMTP Port: 587 (TLS) or 465 (SSL)</li>
                                                                            <li>Note: Requires App Password if 2FA is enabled</li>
                                                                        </ul>
                                                                    </li>
                                                                    <li><strong>Outlook/Hotmail:</strong>
                                                                        <ul>
                                                                            <li>SMTP Host: smtp.office365.com</li>
                                                                            <li>SMTP Port: 587 (TLS)</li>
                                                                        </ul>
                                                                    </li>
                                                                    <li><strong>Yahoo Mail:</strong>
                                                                        <ul>
                                                                            <li>SMTP Host: smtp.mail.yahoo.com</li>
                                                                            <li>SMTP Port: 587 (TLS)</li>
                                                                        </ul>
                                                                    </li>
                                                                </ul>
                                                            </div>
                                                        </div>
                                                        <!-- Right Column -->
                                                        <div class="col-md-6">
                                                            <div class="mb-3">
                                                                <strong>Security Tips:</strong>
                                                                <ul class="mb-3">
                                                                    <li>Always use TLS or SSL encryption</li>
                                                                    <li>For Gmail, use App Password instead of your regular password</li>
                                                                    <li>Keep your SMTP credentials secure</li>
                                                                </ul>
                                                                <strong>Testing:</strong>
                                                                <ul class="mb-0">
                                                                    <li>Use the Test Email button to verify your configuration</li>
                                                                    <li>Check your spam folder if test emails are not received</li>
                                                                    <li>Ensure your email provider allows SMTP access</li>
                                                                </ul>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <div class="mb-3">
                                                            <label class="form-label">SMTP Host</label>
                                                            <input type="text" class="form-control" name="smtp_host" 
                                                                   value="<?php echo htmlspecialchars($settings['smtp_host'] ?? ''); ?>" required>
                                                            <small class="text-muted">Example: smtp.gmail.com</small>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <div class="mb-3">
                                                            <label class="form-label">SMTP Port</label>
                                                            <input type="text" class="form-control" name="smtp_port" 
                                                                   value="<?php echo htmlspecialchars($settings['smtp_port'] ?? ''); ?>" required>
                                                            <small class="text-muted">Common ports: 587 (TLS) or 465 (SSL)</small>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <div class="mb-3">
                                                            <label class="form-label">SMTP Username</label>
                                                            <input type="text" class="form-control" name="smtp_username" 
                                                                   value="<?php echo htmlspecialchars($settings['smtp_username'] ?? ''); ?>" required>
                                                            <small class="text-muted">Your full email address</small>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <div class="mb-3">
                                                            <label class="form-label">SMTP Password</label>
                                                            <input type="password" class="form-control" name="smtp_password" 
                                                                   value="<?php echo htmlspecialchars($settings['smtp_password'] ?? ''); ?>" required>
                                                            <small class="text-muted">For Gmail, use App Password if 2FA is enabled</small>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <div class="mb-3">
                                                            <label class="form-label">SMTP Encryption</label>
                                                            <select class="form-select" name="smtp_encryption" required>
                                                                <option value="tls" <?php echo ($settings['smtp_encryption'] ?? '') === 'tls' ? 'selected' : ''; ?>>TLS</option>
                                                                <option value="ssl" <?php echo ($settings['smtp_encryption'] ?? '') === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                                                            </select>
                                                            <small class="text-muted">TLS is recommended for most providers</small>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <div class="mb-3">
                                                            <label class="form-label">Sender Name</label>
                                                            <input type="text" class="form-control" name="smtp_sender_name" 
                                                                   value="<?php echo htmlspecialchars($settings['smtp_sender_name'] ?? 'POS System'); ?>" required>
                                                            <small class="text-muted">Name that will appear in recipient's inbox</small>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="mb-3">
                                                    <label class="form-label">Test Email Address</label>
                                                    <div class="input-group">
                                                        <input type="email" class="form-control" name="email_test_address" 
                                                               value="<?php echo htmlspecialchars($settings['email_test_address'] ?? ''); ?>" required>
                                                        <button type="submit" name="test_email" class="btn btn-info" data-permission="manage_email_settings" data-permission-label="Manage Email Settings">
                                                            <i class="fas fa-paper-plane"></i> Test Email
                                                        </button>
                                                    </div>
                                                    <small class="text-muted">Enter an email address to test the configuration</small>
                                                </div>
                                            </div>

                                            <div class="text-end">
                                                <button type="submit" name="save_email_settings" class="btn btn-primary" data-permission="manage_email_settings" data-permission-label="Manage Email Settings">
                                                    <i class="fas fa-save"></i> Save Email Settings
                                                </button>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>

                            <!-- M-Pesa Settings -->
                            <div class="tab-pane fade" id="mpesa" role="tabpanel">
                                <div class="settings-group">
                                    <h3>M-Pesa Integration Settings</h3>
                                    <?php if ($success && isset($_POST['save_mpesa_settings'])): ?>
                                        <div class="alert alert-success"><?php echo $success; ?></div>
                                    <?php endif; ?>
                                    <?php if ($error && isset($_POST['save_mpesa_settings'])): ?>
                                        <div class="alert alert-danger"><?php echo $error; ?></div>
                                    <?php endif; ?>
                                    <form method="post" id="mpesaSettingsForm">
                                        <div class="form-check form-switch mb-3">
                                            <input class="form-check-input" type="checkbox" id="enableMpesa" name="enable_mpesa" value="1" 
                                                   <?php if (($settings['enable_mpesa'] ?? '0') == '1') echo 'checked'; ?>>
                                            <label class="form-check-label" for="enableMpesa">Enable M-Pesa Payments</label>
                                        </div>
                                        <div id="mpesaCredentials" class="<?php echo ($settings['enable_mpesa'] ?? '0') == '1' ? '' : 'd-none'; ?>">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Daraja Consumer Key</label>
                                                        <input type="text" class="form-control" name="mpesa_consumer_key" 
                                                               value="<?php echo htmlspecialchars($settings['mpesa_consumer_key'] ?? ''); ?>"
                                                               required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Daraja Consumer Secret</label>
                                                        <input type="text" class="form-control" name="mpesa_consumer_secret" 
                                                               value="<?php echo htmlspecialchars($settings['mpesa_consumer_secret'] ?? ''); ?>"
                                                               required>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Shortcode</label>
                                                        <input type="text" class="form-control" name="mpesa_shortcode" 
                                                               value="<?php echo htmlspecialchars($settings['mpesa_shortcode'] ?? ''); ?>"
                                                               required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label">Passkey</label>
                                                        <input type="text" class="form-control" name="mpesa_passkey" 
                                                               value="<?php echo htmlspecialchars($settings['mpesa_passkey'] ?? ''); ?>"
                                                               required>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <button type="submit" name="save_mpesa_settings" class="btn btn-success" data-permission="manage_mpesa_settings" data-permission-label="Manage M-Pesa Settings">Save M-Pesa Settings</button>
                                    </form>
                                </div>
                            </div>

                            <!-- SMS Settings -->
                            <div class="tab-pane fade" id="sms" role="tabpanel">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0">SMS Settings</h5>
                                    </div>
                                    <div class="card-body">
                                        <form method="post" action="">
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" id="enableSMS" name="enable_sms" value="1" 
                                                           <?php echo ($settings['enable_sms'] ?? '0') === '1' ? 'checked' : ''; ?>>
                                            <label class="form-check-label" for="enableSMS">Enable SMS Notifications</label>
                                                </div>
                                        </div>
                                        
                                            <div id="smsCredentials" class="<?php echo ($settings['enable_sms'] ?? '0') === '1' ? '' : 'd-none'; ?>">
                                            <div class="mb-3">
                                                <label class="form-label">SMS Provider</label>
                                                <select class="form-select" name="sms_provider" id="smsProvider" required>
                                                    <option value="">Select Provider</option>
                                                        <option value="africas_talking" <?php echo ($settings['sms_provider'] ?? '') === 'africas_talking' ? 'selected' : ''; ?>>Africa's Talking</option>
                                                        <option value="twilio" <?php echo ($settings['sms_provider'] ?? '') === 'twilio' ? 'selected' : ''; ?>>Twilio</option>
                                                </select>
                                            </div>
                                            
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">API Key</label>
                                                    <input type="text" class="form-control" name="settings[sms_api_key]" 
                                                               value="<?php echo htmlspecialchars($settings['sms_api_key'] ?? ''); ?>"
                                                           required>
                                                    <small class="form-text text-muted">For Africa's Talking: API Key<br>For Twilio: Account SID</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">API Secret</label>
                                                    <input type="password" class="form-control" name="settings[sms_api_secret]" 
                                                               value="<?php echo htmlspecialchars($settings['sms_api_secret'] ?? ''); ?>"
                                                           required>
                                                    <small class="form-text text-muted">For Africa's Talking: API Secret<br>For Twilio: Auth Token</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Sender ID</label>
                                                    <input type="text" class="form-control" name="settings[sms_sender_id]" 
                                                               value="<?php echo htmlspecialchars($settings['sms_sender_id'] ?? ''); ?>"
                                                           required>
                                                    <small class="form-text text-muted">For Africa's Talking: Sender ID<br>For Twilio: Twilio Phone Number</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Test Phone Number</label>
                                                    <input type="text" class="form-control" name="settings[sms_test_number]" 
                                                               value="<?php echo htmlspecialchars($settings['sms_test_number'] ?? ''); ?>"
                                                           placeholder="+254XXXXXXXXX">
                                                    <small class="form-text text-muted">Enter a phone number to test SMS configuration</small>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="text-end">
                                            <button type="submit" name="test_sms_settings" class="btn btn-info me-2" data-permission="manage_sms_settings" data-permission-label="Manage SMS Settings">
                                                <i class="fas fa-mobile-alt"></i> Test SMS
                                            </button>
                                            <button type="submit" name="save_sms_settings" class="btn btn-primary" data-permission="manage_sms_settings" data-permission-label="Manage SMS Settings">
                                                <i class="fas fa-save"></i> Save SMS Settings
                                            </button>
                                        </div>
                                    </form>
                                    </div>
                                </div>
                            </div>

                            <!-- Password Management -->
                            <div class="tab-pane fade" id="password" role="tabpanel">
                                <div class="settings-group">
                                    <h3>Change Password</h3>
                                    <form method="post" class="needs-validation" novalidate>
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">Current Password</label>
                                                <input type="password" class="form-control" name="current_password" required>
                                            </div>
                                        </div>
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">New Password</label>
                                                <input type="password" class="form-control" name="new_password" required minlength="8">
                                                <div class="form-text">Password must be at least 8 characters long</div>
                                            </div>
                                        </div>
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">Confirm New Password</label>
                                                <input type="password" class="form-control" name="confirm_password" required minlength="8">
                                            </div>
                                        </div>
                                        <div class="row">
                                            <div class="col-md-6">
                                                <button type="submit" name="change_password" class="btn btn-primary" data-permission="manage_settings" data-permission-label="Manage Settings">
                                                    <i class="fas fa-key"></i> Change Password
                                                </button>
                                            </div>
                                        </div>
                                    </form>
                                </div>
                            </div>

                            <!-- Payment Settings -->
                            <div class="tab-pane fade" id="payment" role="tabpanel">
                                <div class="settings-group">
                                    <div class="settings-group-title">
                                        <i class="fas fa-cash-register"></i>
                                        Cash Drawer Settings
                        </div>
                                    <form method="post" id="cashDrawerForm">
                                        <div class="row">
                                            <div class="col-md-6 border-end pe-4">
                                                <div class="mb-4">
                                                    <label class="form-label">Default Cash Drawer</label>
                                                    <select class="form-select" 
                                                            name="default_cash_drawer"
                                                            data-bs-toggle="tooltip"
                                                            data-bs-placement="top"
                                                            title="Last saved: <?php echo isset($storeSettings['default_cash_drawer']) ? ucfirst($storeSettings['default_cash_drawer']) . ' Cash Drawer' : 'Not set'; ?>">
                                                        <option value="main" <?php echo ($storeSettings['default_cash_drawer'] ?? '') === 'main' ? 'selected' : ''; ?>>Main Cash Drawer</option>
                                                        <option value="secondary" <?php echo ($storeSettings['default_cash_drawer'] ?? '') === 'secondary' ? 'selected' : ''; ?>>Secondary Cash Drawer</option>
                                                    </select>
                                                    <div class="settings-description">Select the default cash drawer for transactions</div>
                    </div>
                                                <div class="mb-4">
                                                    <label class="form-label">Opening Balance</label>
                                                    <div class="input-group">
                                                        <span class="input-group-text"><?php echo $currency; ?></span>
                                                        <input type="number" 
                                                               class="form-control" 
                                                               name="opening_balance" 
                                                               value="<?php echo isset($storeSettings['opening_balance']) ? (float)$storeSettings['opening_balance'] : '0.00'; ?>" 
                                                               step="0.01" 
                                                               min="0"
                                                               data-bs-toggle="tooltip"
                                                               data-bs-placement="top"
                                                               title="Last saved: <?php echo isset($storeSettings['opening_balance']) ? $currency . ' ' . number_format((float)$storeSettings['opening_balance'], 2) : $currency . ' 0.00'; ?>"
                                                               onchange="validateOpeningBalance(this)"
                                                               required>
                </div>
                                                    <div class="settings-description">Starting amount in the cash drawer</div>
                                                    <div class="invalid-feedback" id="openingBalanceError"></div>
            </div>

                                                <script>
                                                function validateOpeningBalance(input) {
                                                    const value = parseFloat(input.value);
                                                    const errorDiv = document.getElementById('openingBalanceError');
                                                    
                                                    if (isNaN(value)) {
                                                        input.classList.add('is-invalid');
                                                        errorDiv.textContent = 'Please enter a valid number';
                                                        return false;
                                                    }
                                                    
                                                    if (value < 0) {
                                                        input.classList.add('is-invalid');
                                                        errorDiv.textContent = 'Opening balance cannot be negative';
                                                        return false;
                                                    }
                                                    
                                                    if (value > 1000000) {
                                                        input.classList.add('is-invalid');
                                                        errorDiv.textContent = 'Opening balance cannot exceed 1,000,000';
                                                        return false;
                                                    }
                                                    
                                                    input.classList.remove('is-invalid');
                                                    errorDiv.textContent = '';
                                                    return true;
                                                }

                                                // Initialize tooltips
                                                document.addEventListener('DOMContentLoaded', function() {
                                                    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
                                                    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                                                        return new bootstrap.Tooltip(tooltipTriggerEl);
                                                    });
                                                });
                                                </script>
                                                <div class="mb-4" id="closingBalanceSection" style="display: none;">
                                                    <label class="form-label">Closing Balance</label>
                                                    <div class="input-group">
                                                        <span class="input-group-text"><?php echo $currency; ?></span>
                                                        <input type="number" 
                                                               class="form-control" 
                                                               name="closing_balance" 
                                                               value="<?php echo isset($storeSettings['closing_balance']) ? (float)$storeSettings['closing_balance'] : '0.00'; ?>" 
                                                               step="0.01" 
                                                               min="0"
                                                               data-bs-toggle="tooltip"
                                                               data-bs-placement="top"
                                                               title="Last saved: <?php echo isset($storeSettings['closing_balance']) ? $currency . ' ' . number_format((float)$storeSettings['closing_balance'], 2) : $currency . ' 0.00'; ?>"
                                                               onchange="validateClosingBalance(this)"
                                                               required>
        </div>
                                                    <div class="settings-description">Ending amount in the cash drawer</div>
                                                    <div class="invalid-feedback" id="closingBalanceError"></div>
    </div>

                                                <script>
                                                // Existing validateOpeningBalance function...

                                                function validateClosingBalance(input) {
                                                    const value = parseFloat(input.value);
                                                    const errorDiv = document.getElementById('closingBalanceError');
                                                    const openingBalance = parseFloat(document.querySelector('input[name="opening_balance"]').value) || 0;
                                                    
                                                    if (isNaN(value)) {
                                                        input.classList.add('is-invalid');
                                                        errorDiv.textContent = 'Please enter a valid number';
                                                        return false;
                                                    }
                                                    
                                                    if (value < 0) {
                                                        input.classList.add('is-invalid');
                                                        errorDiv.textContent = 'Closing balance cannot be negative';
                                                        return false;
                                                    }
                                                    
                                                    if (value > 1000000) {
                                                        input.classList.add('is-invalid');
                                                        errorDiv.textContent = 'Closing balance cannot exceed 1,000,000';
                                                        return false;
                                                    }

                                                    // Calculate expected amount (opening balance + total sales)
                                                    const totalSales = parseFloat(document.querySelector('.drawer-amount').textContent.replace(/[^0-9.-]+/g, '')) || 0;
                                                    const expectedAmount = openingBalance + totalSales;
                                                    const difference = Math.abs(value - expectedAmount);
                                                    
                                                    if (difference > 100) { // Allow for small discrepancies
                                                        input.classList.add('is-invalid');
                                                        errorDiv.textContent = `Warning: Expected amount is ${$currency} ${expectedAmount.toFixed(2)}. Difference: ${$currency} ${difference.toFixed(2)}`;
                                                        return false;
                                                    }
                                                    
                                                    input.classList.remove('is-invalid');
                                                    errorDiv.textContent = '';
                                                    return true;
                                                }

                                                // End Day button click handler - protected with timeout
                                                setTimeout(() => {
                                                    try {
                                                        const endDayBtn = document.getElementById('endDayBtn');
                                                        if (endDayBtn) {
                                                            endDayBtn.addEventListener('click', function() {
                                                    const openingBalance = parseFloat(document.querySelector('input[name="opening_balance"]').value) || 0;
                                                    const totalSales = parseFloat(document.querySelector('.drawer-amount').textContent.replace(/[^0-9.-]+/g, '')) || 0;
                                                    const expectedAmount = openingBalance + totalSales;

                                                    if (confirm(`Are you sure you want to end the day?\n\nOpening Balance: ${$currency} ${openingBalance.toFixed(2)}\nTotal Sales: ${$currency} ${totalSales.toFixed(2)}\nExpected Amount: ${$currency} ${expectedAmount.toFixed(2)}`)) {
                                                        document.getElementById('closingBalanceSection').style.display = 'block';
                                                        // Focus on closing balance input
                                                        document.querySelector('input[name="closing_balance"]').focus();
                                                    }
                                                });
                                                        }
                                                    } catch (error) {
                                                        console.error('Error setting up End Day button:', error);
                                                    }
                                                }, 50);

                                                // Form submission validation - protected with timeout
                                                setTimeout(() => {
                                                    try {
                                                        const cashDrawerForm = document.getElementById('cashDrawerForm');
                                                        if (cashDrawerForm) {
                                                            cashDrawerForm.addEventListener('submit', function(e) {
                                                    const closingBalanceSection = document.getElementById('closingBalanceSection');
                                                    if (closingBalanceSection.style.display === 'block') {
                                                        const closingBalance = document.querySelector('input[name="closing_balance"]');
                                                        if (!validateClosingBalance(closingBalance)) {
                                                            e.preventDefault();
                                                            closingBalance.focus();
                                                        }
                                                    }
                                                });
                                                        }
                                                    } catch (error) {
                                                        console.error('Error setting up cash drawer form validation:', error);
                                                    }
                                                }, 50);
                                                </script>
                                            </div>
                                            <div class="col-md-6 ps-4">
                                                <div class="mb-4">
                                                    <label class="form-label">Cash Drawer Status</label>
                                                    <div class="d-flex align-items-center">
                                                        <div class="form-check form-switch">
                                                            <input class="form-check-input" type="checkbox" id="drawer_status" name="drawer_status" <?php echo ($storeSettings['drawer_status'] ?? '') === 'open' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="drawer_status">Drawer is <span id="drawerStatusText"><?php echo ($storeSettings['drawer_status'] ?? '') === 'open' ? 'open' : 'closed'; ?></span></label>
                                                        </div>
                                                        <span class="drawer-status <?php echo ($storeSettings['drawer_status'] ?? '') === 'open' ? 'open' : 'closed'; ?>" id="drawerStatusBadge">
                                                            <i class="fas fa-circle me-2"></i><span id="drawerStatusLabel"><?php echo ($storeSettings['drawer_status'] ?? '') === 'open' ? 'Open' : 'Closed'; ?></span>
                                                        </span>
                                                    </div>
                                                    <div class="settings-description">Current status of the cash drawer</div>
                                                </div>
                                                <div class="mb-4">
                                                    <label class="form-label">Total Sales (Today, Cash)</label>
                                                    <div class="drawer-amount">
                                                        <?php 
                                                        $today = date('Y-m-d');
                                                        $stmt = $pdo->prepare("SELECT COALESCE(SUM(total_amount),0) as total_sales_amount FROM sales WHERE DATE(created_at) = ? AND payment_method = 'cash' AND status = 'completed'");
                                                        $stmt->execute([$today]);
                                                        $today_cash_sales = $stmt->fetchColumn();
                                                        echo $currency . ' ' . number_format($today_cash_sales, 2);
                                                        ?>
                                                    </div>
                                                    <div class="settings-description">Total cash sales for today</div>
                                                </div>
                                                <div class="drawer-actions mb-4">
                                                    <button type="button" class="btn btn-primary" id="reconcileDrawerBtn">
                                                        <i class="fas fa-sync-alt me-2"></i>Reconcile Drawer
                                                    </button>
                                                    <button type="button" class="btn btn-outline-primary" id="viewHistoryBtn">
                                                        <i class="fas fa-history me-2"></i>View History
                                                    </button>
                                                    <button type="button" class="btn btn-warning" id="endDayBtn">
                                                        <i class="fas fa-clock me-2"></i>End Day
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="mt-4">
                                            <button type="submit" class="btn btn-primary" name="save_settings" data-permission="manage_settings" data-permission-label="Manage Settings">
                                                <i class="fas fa-save"></i> Save Settings
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>

                            <!-- WhatsApp Settings -->
                            <div class="tab-pane fade" id="whatsapp" role="tabpanel">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0"><i class="fab fa-whatsapp me-2"></i>WhatsApp Settings</h5>
                                    </div>
                                    <div class="card-body">
                                        <form method="post" id="whatsappSettingsForm">
                                            <div class="form-check form-switch mb-3">
                                                <input class="form-check-input" type="checkbox" id="enableWhatsapp" name="enable_whatsapp" value="1" <?php echo ($settings['enable_whatsapp'] ?? '0') === '1' ? 'checked' : ''; ?>>
                                                <label class="form-check-label" for="enableWhatsapp">Enable WhatsApp Reminders</label>
                                            </div>
                                            
                                            <div id="whatsappSettings" class="<?php echo ($settings['enable_whatsapp'] ?? '0') === '1' ? '' : 'd-none'; ?>">
                                                <div class="mb-3">
                                                    <label class="form-label"> Select WhatsApp Method</label>
                                                    <select class="form-control" name="whatsapp_method" id="whatsappMethod">
                                                        <option value="web_link" <?php echo ($settings['whatsapp_method'] ?? 'web_link') === 'web_link' ? 'selected' : ''; ?>>WhatsApp Web Links (No API Required)</option>
                                                        <option value="business_app" <?php echo ($settings['whatsapp_method'] ?? 'web_link') === 'business_app' ? 'selected' : ''; ?>>WhatsApp Business App</option>
                                                        <option value="api" <?php echo ($settings['whatsapp_method'] ?? 'web_link') === 'api' ? 'selected' : ''; ?>>WhatsApp Business API</option>
                                                    </select>
                                                    <small class="form-text text-muted">
                                                        <strong>Web Links:</strong> Generate WhatsApp links for manual sending (Recommended)<br>
                                                        <strong>Business App:</strong> Use WhatsApp Business app integration<br>
                                                        <strong>API:</strong> Use official WhatsApp Business API (requires setup)
                                                    </small>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">WhatsApp Test Number</label>
                                                    <input type="text" class="form-control" name="whatsapp_test_number" id="whatsappTestNumber" value="<?php echo htmlspecialchars($settings['whatsapp_test_number'] ?? ''); ?>" placeholder="e.g. +2547XXXXXXXX">
                                                    <small class="form-text text-muted">Enter a test number to verify WhatsApp integration</small>
                                                </div>
                                                
                                                <!-- API Settings (only shown when API method is selected) -->
                                                <div id="apiSettings" class="mb-3 <?php echo ($settings['whatsapp_method'] ?? 'web_link') === 'api' ? '' : 'd-none'; ?>">
                                                    <div class="alert alert-info">
                                                        <i class="fas fa-info-circle me-2"></i>
                                                        <strong>WhatsApp Business API Setup:</strong> You need to set up a WhatsApp Business API account and get your credentials.
                                                    </div>
                                                    <div class="mb-3">
                                                        <label class="form-label">WhatsApp API Key</label>
                                                        <input type="text" class="form-control" name="whatsapp_api_key" id="whatsappApiKey" value="<?php echo htmlspecialchars($settings['whatsapp_api_key'] ?? ''); ?>" placeholder="Enter WhatsApp API Key">
                                                        <small class="form-text text-muted">Enter your WhatsApp Business API Key</small>
                                                    </div>
                                                    <div class="mb-3">
                                                        <label class="form-label">WhatsApp API URL</label>
                                                        <input type="text" class="form-control" name="whatsapp_api_url" id="whatsappApiUrl" value="<?php echo htmlspecialchars($settings['whatsapp_api_url'] ?? ''); ?>" placeholder="https://graph.facebook.com/v18.0">
                                                        <small class="form-text text-muted">Enter your WhatsApp Business API URL</small>
                                                    </div>
                                                    <div class="mb-3">
                                                        <label class="form-label">WhatsApp Phone Number ID</label>
                                                        <input type="text" class="form-control" name="whatsapp_phone_number_id" id="whatsappPhoneNumberId" value="<?php echo htmlspecialchars($settings['whatsapp_phone_number_id'] ?? ''); ?>" placeholder="Enter WhatsApp Phone Number ID">
                                                        <small class="form-text text-muted">Enter your WhatsApp Phone Number ID</small>
                                                    </div>
                                                </div>
                                                
                                                <!-- Business App Settings (only shown when business_app method is selected) -->
                                                <div id="businessAppSettings" class="mb-3 <?php echo ($settings['whatsapp_method'] ?? 'web_link') === 'business_app' ? '' : 'd-none'; ?>">
                                                    <div class="alert alert-info">
                                                        <i class="fas fa-info-circle me-2"></i>
                                                        <strong>WhatsApp Business App:</strong> Use your WhatsApp Business app to send messages manually.
                                                    </div>
                                                    <div class="mb-3">
                                                        <label class="form-label">Business Phone Number</label>
                                                        <input type="text" class="form-control" name="whatsapp_business_phone" id="whatsappBusinessPhone" value="<?php echo htmlspecialchars($settings['whatsapp_business_phone'] ?? ''); ?>" placeholder="e.g. +2547XXXXXXXX">
                                                        <small class="form-text text-muted">Enter your WhatsApp Business phone number</small>
                                                    </div>
                                                </div>
                                                
                                                <!-- Web Link Settings (shown for web_link method) -->
                                                <div id="webLinkSettings" class="mb-3 <?php echo ($settings['whatsapp_method'] ?? 'web_link') === 'web_link' ? '' : 'd-none'; ?>">
                                                    <div class="alert alert-success">
                                                        <i class="fas fa-check-circle me-2"></i>
                                                        <strong>WhatsApp Web Links:</strong> No API setup required! The system will generate WhatsApp links that automatically open in a new tab when notifications are sent.
                                                    </div>
                                                    <div class="alert alert-info">
                                                        <i class="fas fa-info-circle me-2"></i>
                                                        <strong>How it works:</strong> When a return is processed, the system will automatically open WhatsApp with the pre-filled message. You just need to click "Send" in WhatsApp.
                                                    </div>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <button type="button" class="btn btn-info" id="testWhatsAppBtn">
                                                        <i class="fas fa-paper-plane me-2"></i>Test WhatsApp Configuration
                                                    </button>
                                                    <small class="form-text text-muted d-block mt-1">Click to send a test WhatsApp message to verify your configuration</small>
                                                </div>
                                            </div>
                                            
                                            <button type="submit" name="save_whatsapp_settings" class="btn btn-success" data-permission="manage_settings" data-permission-label="Manage Settings"><i class="fas fa-save me-2"></i>Save WhatsApp Settings</button>
                                        </form>
                                    </div>
                                </div>
                            </div>

                            <!-- Toggle for Receipts -->
                            <div class="tab-pane fade" id="receipt" role="tabpanel">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0"><i class="fas fa-receipt me-2"></i>Toggle for Receipts</h5>
                                    </div>
                                    <div class="card-body">
                                        <form method="post" id="receiptSettingsForm">
                                            <div class="form-check form-switch mb-3">
                                                <input class="form-check-input" type="checkbox" id="enableUnpaidCreditSection" name="enable_unpaid_credit_section" value="1" <?php echo ($settings['enable_unpaid_credit_section'] ?? '1') === '1' ? 'checked' : ''; ?>>
                                                <label class="form-check-label" for="enableUnpaidCreditSection">Enable Unpaid Credit Section in Sales Receipts</label>
                                            </div>
                                            <div class="alert alert-info">
                                                <i class="fas fa-info-circle me-2"></i>
                                                <strong>Note:</strong> When enabled, the sales receipt will display a section showing all unpaid credit items for the customer. When disabled, this section will be hidden from the receipt.
                                            </div>
                                            <button type="submit" name="save_receipt_settings" class="btn btn-success" data-permission="manage_settings" data-permission-label="Manage Settings">
                                                <i class="fas fa-save me-2"></i>Save Receipt Settings
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Helper function to safely add event listeners
        function safeAddEventListener(elementId, eventType, handler) {
            const element = document.getElementById(elementId);
            if (element) {
                element.addEventListener(eventType, handler);
            }
        }

        // Helper function to safely get element
        function safeGetElement(elementId) {
            return document.getElementById(elementId);
        }

        // Helper function to safely query selector
        function safeQuerySelector(selector) {
            return document.querySelector(selector);
        }

        // Generic toggle function for settings sections
        function setupToggle(checkboxId, sectionId) {
            const checkbox = safeGetElement(checkboxId);
            const section = safeGetElement(sectionId);
            
            // Debug logging to identify missing elements
            if (!checkbox) {
                console.warn(`Checkbox with ID '${checkboxId}' not found`);
                return;
            }
            if (!section) {
                console.warn(`Section with ID '${sectionId}' not found`);
                return;
            }
            
            if (checkbox && section) {
                function toggleSection() {
                    try {
                    if (checkbox.checked) {
                        section.classList.remove('d-none');
                        section.style.display = 'block';
                            // Only set required for visible fields, not all fields in section
                            section.querySelectorAll('input:not([type="hidden"]), select').forEach(input => {
                                // Only set required if the parent container is visible
                                const parentContainer = input.closest('.d-none');
                                if (!parentContainer) {
                            input.disabled = false;
                                    // Don't automatically set required - let method-specific logic handle this
                                }
                        });
                    } else {
                        section.classList.add('d-none');
                        section.style.display = 'none';
                        section.querySelectorAll('input, select').forEach(input => {
                            input.required = false;
                            input.disabled = true;
                        });
                        }
                    } catch (error) {
                        console.error(`Error in toggleSection for ${checkboxId}:`, error);
                    }
                }
                
                try {
                    if (checkbox && typeof checkbox.addEventListener === 'function') {
                checkbox.addEventListener('change', toggleSection);
                toggleSection(); // Initial call
                    } else {
                        console.error(`Checkbox ${checkboxId} is not a valid element or doesn't have addEventListener method`);
                    }
                } catch (error) {
                    console.error(`Error setting up toggle for ${checkboxId}:`, error);
                }
            }
        }

        // Initialize all toggles with correct IDs
        // Add a small delay to ensure all elements are fully loaded
        setTimeout(() => {
            try {
                console.log('Initializing toggles...');
                
                // Check if all elements exist before setting up toggles
                const togglePairs = [
                    ['enableSMS', 'smsCredentials'],
                    ['enableCreditSystem', 'creditSettings'],
                    ['enableMpesa', 'mpesaCredentials'],
                    ['enable_email', 'emailSettings'],
                    ['enableWhatsapp', 'whatsappSettings']
                ];
                
                togglePairs.forEach(([checkboxId, sectionId]) => {
                    const checkbox = document.getElementById(checkboxId);
                    const section = document.getElementById(sectionId);
                    console.log(`Checking ${checkboxId}: ${checkbox ? 'found' : 'NOT FOUND'}, ${sectionId}: ${section ? 'found' : 'NOT FOUND'}`);
                });
                
        setupToggle('enableSMS', 'smsCredentials');
        setupToggle('enableCreditSystem', 'creditSettings'); // Fixed ID to match HTML
        setupToggle('enableMpesa', 'mpesaCredentials');
        setupToggle('enable_email', 'emailSettings');
        setupToggle('enableWhatsapp', 'whatsappSettings'); // Updated for new WhatsApp structure

                console.log('Toggle initialization completed');
                
                // WhatsApp method switching - moved inside setTimeout for safety
        const whatsappMethod = safeGetElement('whatsappMethod');
        if (whatsappMethod) {
            function toggleWhatsAppMethodFields() {
                const method = whatsappMethod.value;
                const apiSettings = safeGetElement('apiSettings');
                const businessAppSettings = safeGetElement('businessAppSettings');
                const webLinkSettings = safeGetElement('webLinkSettings');
                        const whatsappEnabled = safeGetElement('enableWhatsapp');
                
                        // Only proceed if WhatsApp is enabled
                        const isWhatsappEnabled = whatsappEnabled && whatsappEnabled.checked;
                        
                        // Hide all method-specific settings first
                if (apiSettings) apiSettings.classList.add('d-none');
                if (businessAppSettings) businessAppSettings.classList.add('d-none');
                if (webLinkSettings) webLinkSettings.classList.add('d-none');
                
                // Show relevant settings based on method
                switch (method) {
                    case 'api':
                        if (apiSettings) apiSettings.classList.remove('d-none');
                        break;
                    case 'business_app':
                        if (businessAppSettings) businessAppSettings.classList.remove('d-none');
                        break;
                    case 'web_link':
                        if (webLinkSettings) webLinkSettings.classList.remove('d-none');
                        break;
                }
                
                        // Clear all required attributes first
                        const allFields = ['whatsapp_api_key', 'whatsapp_api_url', 'whatsapp_phone_number_id', 'whatsapp_business_phone', 'whatsapp_test_number'];
                        allFields.forEach(fieldName => {
                            const field = safeQuerySelector(`input[name="${fieldName}"]`);
                            if (field) {
                                field.required = false;
                                field.disabled = true;
                            }
                        });
                        
                        // Only set required and enable fields if WhatsApp is enabled
                        if (isWhatsappEnabled) {
                            // Test number is always required when WhatsApp is enabled
                            const testNumberField = safeQuerySelector('input[name="whatsapp_test_number"]');
                            if (testNumberField) {
                                testNumberField.required = true;
                                testNumberField.disabled = false;
                            }
                            
                            // Method-specific required fields
                const apiFields = ['whatsapp_api_key', 'whatsapp_api_url', 'whatsapp_phone_number_id'];
                const businessFields = ['whatsapp_business_phone'];
                
                            if (method === 'api') {
                apiFields.forEach(fieldName => {
                    const field = safeQuerySelector(`input[name="${fieldName}"]`);
                    if (field) {
                                        field.required = true;
                                        field.disabled = false;
                    }
                });
                            } else if (method === 'business_app') {
                businessFields.forEach(fieldName => {
                    const field = safeQuerySelector(`input[name="${fieldName}"]`);
                    if (field) {
                                        field.required = true;
                                        field.disabled = false;
                    }
                });
                            }
                        }
            }
            
                    try {
            whatsappMethod.addEventListener('change', toggleWhatsAppMethodFields);
                        
                        // Also listen to WhatsApp enable checkbox changes
                        const whatsappEnabled = safeGetElement('enableWhatsapp');
                        if (whatsappEnabled) {
                            whatsappEnabled.addEventListener('change', toggleWhatsAppMethodFields);
                        }
                        
            toggleWhatsAppMethodFields(); // Initial call
                        console.log('WhatsApp method switching initialized successfully');
                    } catch (error) {
                        console.error('Error setting up WhatsApp method switching:', error);
                    }
                } else {
                    console.warn('WhatsApp method selector not found');
                }
                
                // WhatsApp test button functionality
                const testWhatsAppBtn = safeGetElement('testWhatsAppBtn');
                if (testWhatsAppBtn) {
                    // Enhanced WhatsApp opener with fallback URLs for better app compatibility
                    function openWhatsAppWithFallback(primaryUrl, alternativeUrls) {
                        // Try to detect if user is on mobile device
                        const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
                        
                        if (isMobile && alternativeUrls && alternativeUrls.app_url) {
                            // On mobile, first try the whatsapp:// protocol to open the app directly
                            const appUrl = alternativeUrls.app_url;
                            
                            // Create a hidden iframe to try the app URL
                            const iframe = document.createElement('iframe');
                            iframe.style.display = 'none';
                            iframe.src = appUrl;
                            document.body.appendChild(iframe);
                            
                            // Set a timeout to fallback to web URL if app doesn't open
                            setTimeout(() => {
                                document.body.removeChild(iframe);
                                // If app didn't open, try the primary URL
                                window.open(primaryUrl, '_blank');
                            }, 1000);
                            
                            // Also try to open the app URL directly
                            window.location.href = appUrl;
                            
                            // Fallback: if nothing works after 2 seconds, open web version
                            setTimeout(() => {
                                if (alternativeUrls.web_url) {
                                    window.open(alternativeUrls.web_url, '_blank');
                                }
                            }, 2000);
                        } else {
                            // On desktop or if no alternative URLs, use primary URL
                            window.open(primaryUrl, '_blank');
                            
                            // Also provide web.whatsapp.com option for desktop users
                            if (alternativeUrls && alternativeUrls.web_url) {
                                setTimeout(() => {
                                    if (confirm('If WhatsApp didn\'t open, would you like to try WhatsApp Web?')) {
                                        window.open(alternativeUrls.web_url, '_blank');
                                    }
                                }, 3000);
                            }
                        }
                    }
                    
                    testWhatsAppBtn.addEventListener('click', function() {
                        const whatsappTestNumberInput = safeGetElement('whatsappTestNumber');
                        const whatsappMethodSelect = safeGetElement('whatsappMethod');
                        
                        const testNumber = whatsappTestNumberInput ? whatsappTestNumberInput.value : '';
                        const whatsappMethod = whatsappMethodSelect ? whatsappMethodSelect.value : 'web_link';
                        
                        if (!testNumber) {
                            alert('Please enter a test phone number first.');
                            if (whatsappTestNumberInput) whatsappTestNumberInput.focus();
                            return;
                        }
                        
                        this.disabled = true;
                        this.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Testing...';
                        
                        // Send test request
                        fetch('../api/test_whatsapp.php?t=' + Date.now(), {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                test_number: testNumber,
                                method: whatsappMethod
                            })
                        })
                        .then(response => {
                            if (!response.ok) {
                                throw new Error('Network response was not ok');
                            }
                            return response.json();
                        })
                        .then(data => {
                            if (data.success) {
                                if (data.whatsapp_url) {
                                    // For web_link and business_app methods, open the WhatsApp link
                                    if (confirm('✅ WhatsApp test successful!\n\nMethod: ' + (data.method || 'web_link') + '\n\nClick OK to open WhatsApp and send the test message.')) {
                                        openWhatsAppWithFallback(data.whatsapp_url, data.alternative_urls);
                                    }
                                } else {
                                    alert('✅ WhatsApp test successful! Check your phone for the test message.\nMethod: ' + (data.method || 'unknown'));
                                }
                            } else {
                                alert('❌ WhatsApp test failed: ' + (data.message || 'Unknown error'));
                            }
                        })
                        .catch(error => {
                            console.error('WhatsApp test error:', error);
                            alert('❌ Error testing WhatsApp: ' + error.message);
                        })
                        .finally(() => {
                            this.disabled = false;
                            this.innerHTML = '<i class="fas fa-paper-plane me-2"></i>Test WhatsApp Configuration';
                        });
                    });
                    console.log('WhatsApp test button initialized successfully');
                } else {
                    console.warn('WhatsApp test button not found');
                }
                
            } catch (error) {
                console.error('Error initializing toggles:', error);
            }
        }, 100);

        // Credit system specific validation - add delay for safety
        setTimeout(() => {
            try {
        const creditSystemToggle = safeGetElement('enableCreditSystem'); // Fixed ID to match HTML
        const creditForm = safeGetElement('creditSettings'); // Fixed ID to match HTML
        
        if (creditSystemToggle && creditForm) {
            function validateCreditLimits() {
                const defaultLimit = safeQuerySelector('input[name="default_credit_limit"]');
                const maxLimit = safeQuerySelector('input[name="max_credit_limit"]');
                
                if (defaultLimit && maxLimit) {
                    const defaultValue = parseFloat(defaultLimit.value);
                    const maxValue = parseFloat(maxLimit.value);
                    
                    if (defaultValue > maxValue) {
                        defaultLimit.setCustomValidity('Default limit cannot exceed maximum limit');
                    } else {
                        defaultLimit.setCustomValidity('');
                    }
                }
            }

            // Add validation listeners to credit limit inputs
            const defaultLimit = safeQuerySelector('input[name="default_credit_limit"]');
            const maxLimit = safeQuerySelector('input[name="max_credit_limit"]');
            
            if (defaultLimit) defaultLimit.addEventListener('input', validateCreditLimits);
            if (maxLimit) maxLimit.addEventListener('input', validateCreditLimits);
                    
                    console.log('Credit system validation initialized successfully');
                } else {
                    console.warn('Credit system elements not found');
        }
            } catch (error) {
                console.error('Error setting up credit system validation:', error);
            }
        }, 150);

        // Cash Drawer Form validation - add delay for safety
        setTimeout(() => {
            try {
        const cashDrawerForm = safeGetElement('cashDrawerForm');
        if (cashDrawerForm) {
            cashDrawerForm.addEventListener('submit', function(e) {
                const closingBalanceSection = safeGetElement('closingBalanceSection');
                if (closingBalanceSection && closingBalanceSection.style.display === 'block') {
                    const closingBalance = safeQuerySelector('input[name="closing_balance"]');
                    if (closingBalance && !validateClosingBalance(closingBalance)) {
                        e.preventDefault();
                        closingBalance.focus();
                    }
                }
            });
                    console.log('Cash drawer form validation initialized successfully');
                } else {
                    console.warn('Cash drawer form not found');
        }
            } catch (error) {
                console.error('Error setting up cash drawer form validation:', error);
            }
        }, 200);

        // End Day button click handler
        safeAddEventListener('endDayBtn', 'click', function() {
            if (confirm('Are you sure you want to end the day? This will record the closing balance.')) {
                const closingBalanceSection = safeGetElement('closingBalanceSection');
                if (closingBalanceSection) {
                    closingBalanceSection.style.display = 'block';
                    const closingBalanceInput = safeQuerySelector('input[name="closing_balance"]');
                    if (closingBalanceInput) {
                        closingBalanceInput.focus();
                    }
                }
            }
        });

        // Drawer status toggle handler
        safeAddEventListener('drawer_status', 'change', function() {
            const statusText = this.checked ? 'open' : 'closed';
            const statusTextElement = safeGetElement('drawerStatusText');
            const statusLabelElement = safeGetElement('drawerStatusLabel');
            const statusBadgeElement = safeGetElement('drawerStatusBadge');
            
            if (statusTextElement) statusTextElement.textContent = statusText;
            if (statusLabelElement) statusLabelElement.textContent = statusText.charAt(0).toUpperCase() + statusText.slice(1);
            if (statusBadgeElement) statusBadgeElement.className = 'drawer-status ' + statusText;
        });

        // Reconcile Drawer button click handler
        safeAddEventListener('reconcileDrawerBtn', 'click', function() {
            const openingBalanceInput = safeQuerySelector('input[name="opening_balance"]');
            const drawerAmountElement = safeQuerySelector('.drawer-amount');
            
            if (openingBalanceInput && drawerAmountElement) {
                const openingBalance = parseFloat(openingBalanceInput.value) || 0;
                const totalSales = parseFloat(drawerAmountElement.textContent.replace(/[^0-9.-]+/g, '')) || 0;
                const expectedAmount = openingBalance + totalSales;

                const actualAmount = prompt('Enter the actual amount in the drawer:', expectedAmount.toFixed(2));
                if (actualAmount !== null) {
                    const difference = parseFloat(actualAmount) - expectedAmount;
                    alert(`Expected: ${expectedAmount.toFixed(2)}\nActual: ${actualAmount}\nDifference: ${difference.toFixed(2)}`);
                }
            }
        });

        // View History button click handler
        safeAddEventListener('viewHistoryBtn', 'click', function() {
            alert('View History functionality will be implemented here');
        });
    });

    // Form validation functions
    function validateStoreInfo(event) {
        if (!event.target.id === 'storeInfoForm') {
            return true;
        }
        
        const storeTab = document.querySelector('#store');
        if (!storeTab) return true;

        const inputs = storeTab.querySelectorAll('input[type="text"]:not([disabled])');
        for (const input of inputs) {
            if (input.value.trim() === '') {
                alert('Please fill in all fields in the store information form.');
                input.focus();
                return false;
            }
        }
        return true;
    }

    function validateCreditSettings() {
        const form = document.getElementById('creditSettings');
        if (!form) return true;

        const enableCreditSystem = form.querySelector('input[name="enable_credit_system"]')?.checked || false;
        const defaultCreditLimit = form.querySelector('input[name="default_credit_limit"]')?.value;
        const maxCreditLimit = form.querySelector('input[name="max_credit_limit"]')?.value;
        const creditPaymentTerms = form.querySelector('input[name="credit_payment_terms"]')?.value;
        const latePaymentFee = form.querySelector('input[name="late_payment_fee"]')?.value;

        if (enableCreditSystem) {
            if (!defaultCreditLimit || !maxCreditLimit || !creditPaymentTerms || !latePaymentFee) {
                alert('Please fill all fields in the credit settings form.');
                return false;
            }

            if (parseFloat(defaultCreditLimit) > parseFloat(maxCreditLimit)) {
                alert('Default credit limit cannot exceed maximum credit limit.');
                return false;
            }
        }

        return true;
    }

    // Dynamically set required attribute only for visible tab fields
    function setRequiredFields() {
        const emailTab = document.getElementById('email-settings-tab');
        const emailFields = ['smtp_host', 'smtp_port', 'email_test_address'];
        
        emailFields.forEach(function(id) {
            const el = document.getElementsByName(id)[0];
            if (el) {
                if (emailTab && emailTab.classList.contains('active')) {
                    el.setAttribute('required', 'required');
                } else {
                    el.removeAttribute('required');
                }
            }
        });
    }
    </script>
    
    <?php 
    // Add permission modal components
    echo getPermissionModalCSS();
    echo getPermissionModalHTML();
    echo getPermissionModalJS();
    ?>
    <?php echo getUniversalPermissionModalHTML(); ?>
    <?php echo getUniversalPermissionModalJS(); ?>
</body>
</html> 
